# -*- coding: utf-8 -*-
"""
    Résolution numérique d'une équation différentielle d'ordre 2
    avec la fonction odeint de python (scipy).
    Ici, on utilise un pendule simple, avec amortissement.
    Equa diff : thetapointpoint + 2m.wo.thetapoint + wo**2.theta = wo**2.grandtheta
                                2m = 1 / Q
                                theta(t) = exp(-mwot)*...
    On peut évidemment remplacer theta par n'importe quelle autre variable !
"""

#### Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint

#### Initialisations

# wo pulsation propre en rad/s
wo = 1

# coefficient de frottement
m = 1.3              # en modifiant m on modifie la nature du régime transitoire
                     # m > 1 régime apériodique ; m = 1 régime critique : m < 1 régime pseudo périodique
tau = 1/(m*wo)      # constante de temps
Q = 1 /(2*m)        # facteur de qualité
print (Q)



# Conditions initiales
grandtheta = 0                       # modélise une contrainte extérieure constante
theta0 = 10*np.pi/180                # angle initial en radian
thetapoint0 = 0                      # vitesse angulaire initiale
y0 = [theta0, thetapoint0]           # vecteur initial

# Fonction intervenant dans l'équation différentielle vectorielle :
#     dy/dt = f(y, t) avec vecteur y = (theta, theta point)
def f(y, t) :                       # f(vecteur, temps)
    theta, thetapoint = y
    return [thetapoint, -2*m*wo*thetapoint - wo**2*(theta-grandtheta)]

# Données pour l'expérience numérique
n = 1025                    # nombre de points de l'échantillonnage, puissance de 2 + 1
t0 = 0                      # instant initial en s
t1 = 30                     # instant final en s

#### Expérience numérique
t = np.linspace(t0, t1, n)
sol = odeint(f, y0, t)      # solution = liste des vecteurs pour chaque t

#### Tracés
# theta(t) 
plt.figure(1)                                   # fenetrage
plt.plot(t, sol[:, 0], label='m =' + str(m))   #plt. instruction dans matplotlib.pyplot
                                                # syntaxe plot(liste des abscisses,liste ordonnées,...)
                                                # sol = liste des vecteurs solution
                                                # sol[:, 0] = liste des premières coordonnées des vecteurs de sol
                                                # _ : _ intervalle des valeurs de la liste = sous liste
plt.legend(loc='lower right')                   #légende en bas à droite
plt.xlabel('t (s)')                             #légende axe x
plt.ylabel('theta en rad')                            
plt.grid()
plt.show()

