# -*- coding: utf-8 -*-
"""
    Résolution numérique de l'équation de diffusion
    par un schéma explicite de différences finies.
    L'équation est ici, pour T(x, t),
    ∂T/∂t = D.∂2T/∂x2
    T(x, 0) = T0
    T(x, t) = T1
    T(L, t) = T2
    Discrétisation de l'équation de diffusion : T(x, t) = T[i, j] tableau
    partie temporelle :
    ∂T/∂t = (T[i, j+1] - T[i, j]) / dt
    partie spatiale :
    ∂2T/∂x2 = (T[i+1, j] - 2T[i, j] + T[i-1,j]) / dx^2
        d'où
    T[i, j+1] = T[i, j] + c*(T[i+1, j] - 2*T[i, j] + T[i-1, j])
    avec c = D*dt/dx**2 (sans unité)
"""

#### Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt

#### Initialisations

# Données physiques
L = 1.0      # longueur de la barre en m
T0 = - 10.0  # température initiale de la barre en °C
T1 = 100.0   # température à l'extrémité gauche de la barre (x = 0) en °C
T2 = 20.0    # température à l'extrémité droite de la barre (x = L) en °C
D = 1E-5       # coefficient de diffusion en m^2/s
duree = L**2 / (5*D)     # durée de l'expérience en s (choix dépend de la valeur de D = L^2/temps)
print(f"la durée optimale de l'expérience numérique est {duree:.0f} s")  # mise en forme avec 0 chiffre après la voigule à duree

# Pas de discrétisation (choix de démarche)
dx = 0.01     # en m
c = 0.5         # c = D*dt/dx**2, stabilité numérique si c <= 0.5 (sans unité)
dt = c * dx**2 /D # en s
print(f"période d'échantillonage temporel {dt:.2f} s")   # mise en forme avec 2 chiffres après la virgule à T échant

# Tableaux de travail
Nx = int(L/dx)                  # pour l'espace, 
print("nombre d'échantillons spatiaux " + str(Nx))
x = np.linspace(0.0, L, Nx)     # échantillonnage x, tableau

Nt = int(duree/dt)              # pour le temps, t
print ("nombre d'échantillons temporels " + str(Nt))

t = np.linspace(0.0, duree, Nt)
T = np.zeros((Nx, Nt))          # pour la température, T(x, t), tableau de zéros à 2 entrées = matrice
                                # Nx lignes (100) et Nt colonnes (4000)

# Conditions initiales : température initiale de la barre
for i in range(Nx):
    T[i, 0] = T0

# Conditions aux limites : températures aux extrémités de la barre
for k in range(Nt):     
    T[0, k] = T1           
    T[-1, k] = T2


#### Expérience numérique

for j in range(0, Nt-1):                                             # boucle temporelle
    for i in range(1, Nx-1):                                         # boucle spatiale
        T[i, j+1] = T[i, j] + c*(T[i+1, j] - 2*T[i, j] + T[i-1, j])  # schéma explicite de différences finies
                                                                     # de l'équation de la chaleur

#### Tracés

# Création de la figure pour 3 instants différents

plt.xlabel('x (s)')
plt.ylabel('T (°C)')
#plt.title('Evolution de la température de la barre soumise à un échelon de tension à chaque extrémité. A t < 0 Tbarre = - 10°C')

t1 = duree/100 
t2 = duree/2
t3 = duree*0.99   # attention intervalle ouvert à droite

plt.plot(x, T[:, int(t1/dt)], label= f't = {t1:.0f} s')
plt.plot(x, T[:, int(t2/dt)], label= f't = {t2:.0f} s')
plt.plot(x, T[:, int(t3/dt)], label= f't = {t3 :.0f} s')


plt.legend()
plt.show()