# -*- coding: utf-8 -*-==
"""
    Résolution numérique de l'équation de diffusion
    par un schéma explicite de différences finies.
    L'équation est ici, pour T(x, t),
    ∂T/∂t = D.∂2T/∂x2
    T(x, 0) = T0
    T(x, t) = T1
    T(L, t) = T2
    Discrétisation de l'équation de diffusion : T(x, t) = T[i, j)] tableau
    partie temporelle :
    ∂T/∂t = (T[i, j+1] - T[i, j]) / dt
    partie spatiale
    ∂T/∂x = (T[i+1, j] - T[i, j]) / dx
    ∂2T/∂x2 = ∂(∂T/∂x)/∂x = [(∂T/∂x)(i+1, j) - (∂T/∂x)(i, j)] / dx
    = [(T[i+2, j] - T[i+1, j]) / dx - (T[i+1, j] - T[i, j]) / dx] /dx]
    = (T[i+2, j] - 2T[i+1, j] + T[i, j]) / dx^2
"""

# Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt
from matplotlib import animation

# Initialisations

## Données physiques
L = 1.0     # longueur de la barre
duree = 20000 # durée de l'expérience
T0 = -10.0   # température initiale de la barre
T1 = 100.0  # température à l'extrémité gauche de la barre (x = 0)
T2 = 20.0   # température à l'extrémité droite de la barre (x = L)
D = 1e-5    # coefficient de diffusion en m^2/s

## Pas de discrétisation
dx = 0.01
dt = 5.0

## Tableaux de travail
Nx = int(L/dx)                  # pour l'espace, x
x = np.linspace(0.0, L, Nx)     # échantillonnage x, tableau
Nt = int(duree/dt)              # pour le temps, t
t = np.linspace(0.0, duree, Nt)
T = np.zeros((Nx, Nt))          # pour la température, T(x, t), tableau à 2 entrées = matrice

## Conditions initiales : température initiale de la barre
for i in range(1, Nx-1):
    T[i, 0] = T0

## Conditions aux limites : températures aux extrémités de la barre
for k in range(0, Nt):
    T[0, k] = T1
    T[-1, k] = T2


# Expérience numérique

c = D*dt/dx**2

## Vérification du critère de stabilité du schéma
# if c > 0.5:
#     print('Attention, le schéma est instable : dt doit être inférieur à ', dx**2/(2.0*D))

for j in range(0, Nt-1):            # boucle temporelle
    for i in range(1, Nx-1):        # boucle spatiale
        # schéma explicite de différences finies de l'équation de la chaleur
        T[i, j+1] = T[i, j] + c*(T[i+1, j] - 2*T[i, j] + T[i-1, j])

# Tracés

## Création du réseau de courbes

### Première fenêtre
fig1 = plt.figure(1)
ax1 = plt.axes(xlim=(x[0]-(x[-1]-x[0])/10, x[-1]+(x[-1]-x[0])/10),
               ylim=(T0-(T1-T0)/10, T1+(T1-T0)/10))
ax1.set_xlabel('x')
ax1.set_ylabel('T')

### Tracé des diverses courbes
for k in range(10):
    ax1.plot(x, T[:, int(k*Nt/10)], 'r', lw=2)

## Animation

### Seconde fenêtre
fig2 = plt.figure(2)
ax2 = plt.axes(xlim=(x[0]-(x[-1]-x[0])/10, x[-1]+(x[-1]-x[0])/10),
              ylim=(T0-(T1-T0)/10, T1+(T1-T0)/10))
ax2.set_xlabel('x')
ax2.set_ylabel('T')

### Création d'une courbe vide
courbe, = ax2.plot([], [], 'r', lw=2)

### Fonction d'initialisation de l'animation
def init():
    courbe.set_data([], [])
    return courbe,

### Fonction d'animation proprement dite, appelée à chaque image
def animate(i):
    # remplissage de la courbe
    courbe.set_data(x, T[:, i*20])
    return courbe,

### Création de l'animation
anim = animation.FuncAnimation(fig2, animate, init_func=init,
                               frames=Nt//20, interval=15)


plt.show()
